/**********************************************************************/
/*                               L E D C                              */
/*--------------------------------------------------------------------*/
/*    Description    : Sets the various bits in the BIOS keyboard     */
/*                     flag, causing the LEDs on the AT keyboard to   */
/*                     flash.                                         */
/*--------------------------------------------------------------------*/
/*    Author         : MICHAEL TISCHER                                */
/*    developed on   : 22.08.1988                                     */
/*    last update    : 22.08.1988                                     */
/*--------------------------------------------------------------------*/
/*    (MICROSOFT C)                                                   */
/*    creation       : CL /AS LEDC.C                                  */
/*    call           : LEDC                                           */
/*--------------------------------------------------------------------*/
/*    (BORLAND TURBO C)                                               */
/*    creation       : via the command COMPILE / MAKE                 */
/**********************************************************************/

/*== Include files ===================================================*/

#include <dos.h>

/*== Macros ==========================================================*/

#ifndef MK_FP                           /* was MK_FP already defined? */
#define MK_FP(seg, ofs) ((void far *) ((unsigned long) (seg)<<16|(ofs)))
#endif

/*== Constants =======================================================*/

#define SCRL  16                                   /* Scroll Lock bit */
#define NUML  32                                      /* Num Lock bit */
#define CAPL  64                                     /* Caps Lock bit */
#define INS  128                                        /* Insert bit */

/*-- BIOS_KBF creates a pointer to the BIOS keyboard flag ------------*/

#define BIOS_KBF ((unsigned far *) MK_FP(0x40, 0x17))

/***********************************************************************
*  Function         : D E L A Y                                        *
**--------------------------------------------------------------------**
*  Description      : Waits a certain length of time.                  *
*  Input parameters : PAUSE = the number of milliseconds to wait.      *
*  Return value     : none                                             *
*  Info             : Since this function uses the BIOS timer for time *
*                     measurement, the accuracy is limited to about    *
*                     1/60 of a second.                                *
***********************************************************************/

void delay( unsigned pause )
{
 long timer;                 /* stores the timer value to be reached  */
 union REGS inregs,                 /* stores the processor registers */
            outregs;    /* INREGS before, OUTREGS after the intr call */

 inregs.h.ah = 0;                             /* ftn. no.: read timer */
 int86(0x1a, &inregs, &outregs);         /* call BIOS timer interrupt */

 /*- calculate the target time value  and check to see if this    ----*/
 /*- value has been reached.                                      ----*/

 timer = outregs.x.dx + ((long) outregs.x.cx << 16) +
         (pause * 18 + ((pause << 1) / 10)) / 1000;
 do                                                  /* polling loop */
  int86(0x1a, &inregs, &outregs);                /* read timer again */
 while ((outregs.x.dx + ((long) outregs.x.cx << 16)) <= timer);
}

/***********************************************************************
*  Function         : S E T _ F L A G                                  *
**--------------------------------------------------------------------**
*  Description      : Sets individual bits or flags in the BIOS        *
*                     keyboard flag.                                   *
*  Input parameters : FLAG = the bits or flags to be set               *
*  Return value     : none                                             *
***********************************************************************/

void set_flag( unsigned flag )
{
 union REGS regs;                   /* stores the processor registers */

 *BIOS_KBF |= flag;    /* set the specified bits in the keyboard flag */
 regs.h.ah = 1;                       /* ftn. no.: character present? */
 int86(0x16, &regs, &regs);           /* call BIOS keyboard interrupt */
}

/***********************************************************************
*  Function         : C L R _ F L A G                                  *
**--------------------------------------------------------------------**
*  Description      : Clears individual bits or flags in the BIOS      *
*                     keyboard flag.                                   *
*  Input parameters : FLAG = the bits or flags to be cleared           *
*  Return value     : none                                             *
***********************************************************************/

void clr_flag( unsigned flag )
{
 union REGS regs;                   /* stores the processor registers */

 *BIOS_KBF &= ~flag;          /* mask out bits in the BIOS keyb. flag */
 regs.h.ah = 1;                       /* ftn. no.: character present? */
 int86(0x16, &regs, &regs);           /* call BIOS keyboard interrupt */
}

/**********************************************************************/
/**                           MAIN PROGRAM                           **/
/**********************************************************************/

void main()
{
 unsigned i;                                           /* loop counter*/

 printf("LEDP  -  (c) 1988 by Michael Tischer\n\n");
 printf("Watch the LEDs on your keyboard!\n");

 for (i=0; i<10; ++i)                /* run through the loop 10 times */
  {
   set_flag( CAPL );                                  /* turn CAPS on */
   delay( 100 );                             /* wait 100 milliseconds */
   clr_flag( CAPL );                           /* turn CAPS off again */
   set_flag( NUML);                                    /* turn on NUM */
   delay( 100 );                             /* wait 100 milliseconds */
   clr_flag( NUML );                            /* turn NUM off again */
   set_flag( SCRL);                            /* turn on SCROLL LOCK */
   delay( 100 );                             /* wait 100 milliseconds */
   clr_flag( SCRL );                    /* turn SCROLL LOCK off again */
  }

 for (i=0; i<10; ++i)                /* run through the loop 10 times */
  {
   set_flag(CAPL | SCRL | NUML);                /* all three flags on */
   delay( 200 );                             /* wait 200 milliseconds */
   clr_flag(CAPL | SCRL | NUML);               /* all flags off again */
   delay( 200 );                             /* wait 200 milliseconds */
  }
}
