;**********************************************************************;
;*                             T Y P M B A                            *;
;*--------------------------------------------------------------------*;
;*    Description    : Assembler routine for use with a Turbo Pascal   *;
;*                     program, which sets the key repeat rate of the *;
;*                     AT keyboard.                                   *;
;*--------------------------------------------------------------------*;
;*    Author         : MICHAEL TISCHER                                *;
;*    developed on   : 27.08.1988                                     *;
;*    last update    : 27.08.1988                                     *;
;*--------------------------------------------------------------------*;
;*    to assemble    : MASM TYPMPA;                                   *;
;*                     LINK TYPMPA                                    *;
;*                     EXE2BIN TYPMPA TYPMPA.BIN                      *;
;*                     ... convert to INLINE statements *;
;**********************************************************************;

;== Constants ==========================================================

KB_STATUS_P   equ 64h             ;status port of the keyboard
KB_DATA_P     equ 60h             ;keyboard data port

OB_FULL       equ 1               ;Bit 0 in the keyboard status port
                                  ;one character in the output buffer
IB_FULL       equ 2               ;Bit 1 in the keyboard status port
                                  ;one character in the input buffer

ACK_SIGNAL    equ 0fah            ;keyboard acknowledge signal
SET_TYPEM     equ 0f3h            ;set-key-repeat code

MAX_TRY       equ 3               ;number of retries

;== Program code ========================================================

code      segment para 'CODE'     ;definition of the CODE segment

          org  100h

          assume cs:code, ds:code, ss:code, es:code

;-----------------------------------------------------------------------
;-- SET_TYPM: Determines the key repeat rate to be sent to the  --------
;--           keyboard controller
;-- Info    : Set up as a NEAR call
set_typm   proc near              ;GW expects FAR procedures

sframe0   struc                  ;structure for accessing the stack
bp0        dw ?                   ;stores BP
ret_adr0   dd ?                   ;return address to the caller
                                  ;(FAR address)
trate0      dw ?                   ;address of the var with the rep rate
sframe0    ends                   ;end of the structure

frame      equ [ bp - bp0 ]      ;addresses the elements of the structure

                                  ;The following instructions are executed by Turbo
;           push bp                ;save BP on the stack
;           mov  bp,sp             ;transfer SP to BP

           xor  dl,dl             ;assume transfer failed
           mov  ah,SET_TYPEM      ;set command code for key rep rate
           cli                    ;disable interrupts
           call send_kb           ;send to the controller
           jne  error             ;error? yes --> Error

           mov  ah,byte ptr frame.trate0   ;get address of the TYPRATE variable
           call send_kb           ;send to the controller
           jne  error             ;error? yes --> Error

           inc  dl               ;everything OK, return TRUE

error:     sti                    ;allow interrupts again
           mov  [bp-1],dl           ;put error static there
           pop  bp                ;get BP back from stack
           jmp ende                 ;back to Turbo Pascal
set_typm   endp

;-----------------------------------------------------------------------
;-- SEND_KB: send a byte to the keyboard controller --------------------
;-- Input    : AH = the byte to be sent
;-- Output   : zero flag: 0=error, 1=OK
;-- Registers: AX and the flag register are used
;-- Info     : this routine is intended for use only within this 
;--            module

send_kb   proc near

          push cx                 ;save all registers used in this
          push bx                 ;routine on the stack

          mov  bl,MAX_TRY         ;maximum of MAX_TRY retries

          ;-- wait until the controller is ready to receive data -------

skb_1:    xor  cx,cx              ;maximum of 65536 loop passes
skb_2:    in   al,KB_STATUS_P     ;read contents of the status port
          test al,IB_FULL         ;still a character in the input buffer?
          loopne skb_2            ;yes --> SKB_2

          ;-- send character to the controller -------------------------

          mov  al,ah              ;get character in AL
          out  KB_DATA_P,al       ;send character to the data port
skb_3:    in   al,KB_STATUS_P     ;read contents of the status port
          test al,OB_FULL         ;answer in the output buffer?
          loope skb_3             ;no --> SKB_3

          ;-- get reply from controller and evaluate -------------------

          in   al,KB_DATA_P       ;read reply from data port
          cmp  al,ACK_SIGNAL      ;was the character accepted?
          je   skb_end            ;YES --> everything OK

          ;-- the character was not accepted ---------------------------

          dec  bl                 ;decrement error counter
          jne  skb_2              ;retries left?
                                  ;YES --> SKB_2

          or   bl,1               ;NO, set zero flag to 0, indicating
                                  ;an error

skb_end:  pop  bx                 ;restore the registers from the stack
          pop  cx                 
          ret                     ;back to the caller

send_kb   endp

;----------------------------------------------------------
ende          label near
;== End ===============================================================

code      ends                    ;end of the code segment
          end  set_typm
