/**********************************************************************/
/*                               M E M C                              */
/*--------------------------------------------------------------------*/
/*    Description    : Displays the memory blocks allocated by DOS    */
/*--------------------------------------------------------------------*/
/*    Author         : MICHAEL TISCHER                                */
/*    developed on   : 08/23/1988                                     */
/*    last update    : 05/12/1989                                     */
/*--------------------------------------------------------------------*/
/*    (MICROSOFT C)                                                   */
/*    creation       : CL /AS /Zp memc.c                              */
/*    call           : MEMC                                           */
/*--------------------------------------------------------------------*/
/*    (BORLAND TURBO C)                                               */
/*    creation       : via the Compile-Make command                   */
/*                     (no project file)                              */
/**********************************************************************/

/*== Include files ===================================================*/

#include <dos.h>
#include <stdlib.h>

/*== Typedefs ========================================================*/

typedef unsigned char byte;                 /* build ourselves a byte */
typedef unsigned segadr;                         /* a segment address */
typedef byte boolean;
typedef byte far *FB;                        /* FAR pointer to a byte */

/*== Constants =======================================================*/

#define TRUE  1                    /* needed for working with boolean */
#define FALSE 0

/*== Structures and unions ===========================================*/

struct MCB {                            /* describes an MCB in memory */
            byte     id_code;     /* 'M' = a block follows, 'Z' = end */
            segadr   psp;               /* segment address of the PSP */
            unsigned distance;        /* number of paragraphs reserved */
           };

typedef struct MCB far *MCBPtr;              /* FAR pointer to an MCB */

/*== Macros ==========================================================*/

#ifndef MK_FP                           /* was MK_FP defined already? */
#define MK_FP(seg, ofs) ((void far *) ((unsigned long) (seg)<<16|(ofs)))
#endif


/******************************************************************
*****
*  Function         : F I R S T _ M C B                                *
**--------------------------------------------------------------------**
*  Description      : Returns a pointer to the first MCB.              *
*  Input parameters : none                                             *
*  Return value     : Pointer to the first MCB                         *
*******************************************************************
****/

MCBPtr first_mcb()
{
 union REGS regs;                   /* stores the processor registers */
 struct SREGS sregs;                  /* stores the segment registers */

 regs.h.ah = 0x52;     /* ftn. no.: get address of the DOS info block */
 intdosx( &regs, &regs, &sregs );          /* call DOS interrupt 0x21 */

 /*-- ES:(BX-4) points to the firs MCB, create pointer ---------------*/

 return( *((MCBPtr far *) MK_FP( sregs.es-1, regs.x.bx+12 )) );
}

/***********************************************************************
*  Function         : D U M P                                          *
**--------------------------------------------------------------------**
*  Description      : Outputs hex and ASCII dump of a memory range.    *
*  Input parameters : - bptr : pointer to a memory area                *
*                     - num  : number of dump lines (each 16 bytes)    *
*  Return value     : none                                             *
***********************************************************************/

void dump( FB bptr, byte num)
{
 FB lptr;                 /* running pointer for printing a dump line */
 unsigned offset;                  /* offset address relative to BPTR */
 byte i;                                              /* loop counter */

 printf("\nDUMP  0123456789ABCDEF        00 01 02 03 04 05 06 07 08");
 printf(" 09 0A 0B 0C 0D 0E 0F\n");
 printf("");
 printf("\n");

 for (offset=0;  num-- ; offset += 16, bptr += 16)
  {                                 /* run through the loop NUM times */
   printf("%04x  ", offset);
   for (lptr=bptr, i=16; i-- ; ++lptr)    /* print character as ASCII */
    printf("%c", (*lptr<32) ? ' ' : *lptr);
   printf("        ");
   for (lptr=bptr, i=16; i-- ; )           /* output character as hex */
    printf("%02X ", *lptr++);
   printf("\n");                             /* move to the next line */
  }
}


/***********************************************************************
*  Function         : T R A C E _ M C B                                *
**--------------------------------------------------------------------**
*  Description      : Traces the chain of MCB's.                       *
*  Input parameters : none                                             *
*  Return value     : none                                             *
***********************************************************************/

void trace_mcb()
{
 static char fenv[] = {                   /* first environment string */
                       'C', 'O', 'M', 'S', 'P', 'E', 'C', '='
                      };

 MCBPtr cur_mcb;                        /* pointer to the current MCB */
 boolean done;                      /* TRUE if the last MCB was found */
 byte nr_mcb,                            /* number of the current MCB */
      i;                                             /* loop variable */
 FB lptr;                       /* running pointer in the environment */

 done = FALSE;                                    /* now we get going */
 nr_mcb = 1;                             /* the first MCB is number 1 */
 cur_mcb = first_mcb();               /* get pointer to the first MCB */
 do                                   /* process the individual MCB's */
  {
   if ( cur_mcb->id_code == 'Z' )                /* last MCB reached? */
    done = TRUE;                                               /* yes */
   printf("MCB number    = %d\n", nr_mcb++);
   printf("MCB address   = %Fp\n", cur_mcb);
   printf("Memory addr.  = %Np:0000\n", FP_SEG(cur_mcb)+1);
   printf("ID            = %c\n", cur_mcb->id_code);
   printf("PSP address   = %Fp\n", (FB) MK_FP(cur_mcb->psp, 0) );
   printf("Size          = %u paragraphs ( %lu bytes )\n",
          cur_mcb->distance, (unsigned long) cur_mcb->distance << 4);
   printf("Contents      = ");

   /*-- is it an environment? -----------------------------------------*/

   for (i=0, lptr=(FB)cur_mcb+16;/* compare first ENV string with FENV */
        ( i<sizeof fenv ) && ( *(lptr++) == fenv[i++] ) ; )
    ;
   if ( i == sizeof fenv )                      /* was a string found? */
    {                                      /* yes, it's an environment */
     printf("environment\n");
     if ( _osmajor >= 3 )                /* DOS version 3.0 or higher? */
      {                                       /* yes, get program name */
       printf("Program name  = ");
       for ( ; !(*(lptr++)==0 && *lptr==0) ; )
        ;                                      /* find last ENV string */
       if ( *(lptr += 3) )            /* is there a program name here? */
        {                                                       /* yes */
         for ( ; *lptr ; )             /* run through the program name */
          printf( "%c", *(lptr++) );             /* output a character */
        }
       else                               /* no program name was found */
        printf("unknown");
       printf("\n");                          /* move to the next line */
      }

     /*-- output the environment strings ------------------------------*/

     printf("Environment strings\n");
     for (lptr=(FB) cur_mcb +16; *lptr ; ++lptr)
      {                                             /* output a string */
       printf("           ");
       for ( ; *lptr ; )  /* run through the string to a NUL character */
        printf( "%c", *(lptr++) );               /* output a character */
       printf("\n");                          /* move to the next line */
      }
    }
   else                                              /* no envrionment */
    {

     /*- is it a PSP? -------------------------------------------------*/
     /*-- (introduced with the command INT 20 (Code=0xCD 0x20)) -------*/

     if (*((unsigned far *) MK_FP( cur_mcb->psp, 0 )) == 0x20cd)
      printf("PSP (with subsequent program)\n");                /* yes */
     else                        /* the command INT 0x20 was not found */
       {
        printf("unidentifiable (program or data)\n");
        dump( (FB) cur_mcb + 16, 5);      /* dump the first 5x16 bytes */
       }
    }

   printf("");
   printf(" Please press a key   \n");
   if ( !done )                                        /* another MCB? */
    {                              /* yes, set pointer to the next MCB */
     cur_mcb = (MCBPtr)
                MK_FP( FP_SEG(cur_mcb) + cur_mcb->distance + 1, 0 );
     getch();                                        /* wait for a key */
    }
  }
 while ( !done );      /* repeat until the last MCB has been processed */
}

/**********************************************************************/
/**                           MAIN PROGRAM                           **/
/**********************************************************************/

void main()

{
 printf("\nMEMC (c) 1988 by Michael Tischer\n\n");
 trace_mcb();                             /* trace the chain of MCB's */
}
