{**********************************************************************}
{*                              T S R P                               *}
{*--------------------------------------------------------------------*}
{*    Description    : creates a TSR program with the help of an      *}
{*                     assembly language module.                      *}
{*--------------------------------------------------------------------*}
{*    Author         : MICHAEL TISCHER                                *}
{*    developed on   : 08/18/1988                                     *}
{*    last update    : 05/26/1989                                     *}
{**********************************************************************}

program TSRP;

uses DOS, CRT;                        { bind in the DOS and CRT units  }

{$M 2048, 0, 5120}       { 2KB for the stack and max. 5KB for the heap }
{$L tsrpa}                              { bind in the assembler module }

const LSHIFT =     1;                                 { left SHIFT key }
      RSHIFT =     2;                                { right SHIFT key }
      CTRL   =     4;                                       { CTRL key }
      ALT    =     8;                                        { ALT key }
      SYSREQ =  1024;                 { SYS REQ key (ST keyboard only) }
      BREAK  =  4096;                                      { BREAK key }
      NUM    =  8192;                                        { NUM key }
      CAPS   = 16384;                                       { CAPS key }
      INSERT = 32768;                                     { INSERT key }

      NO_END_FTN = $FFFF;                 { don't call an end function }

type IdsType = string[ 16 ];     { describes the identification string }
     VBuf    = array[1..25, 1..80] of word;     { describes the screen }
     VPtr    = ^VBuf;                     { pointer to a screen buffer }

var IdString : IdsType;            { the ID string for the TSR program }
    MBuf     : VBuf absolute $B000:0000;    { the monochrome video RAM }
    CBuf     : Vbuf absolute $B800:0000;         { the color video RAM }
    VioPtr   : VPtr;                        { pointer to the video RAM }

{** Declaration of the external functions in the assembly module ******}

procedure TsrInit( PrcPtr   : word;      { offset addr of the TSR proc }
                   KeyMask  : word;           { the hotkey (see CONST) }
                   ResPara  : word;   { number of para. to be reserved }
                   IdString : IdsType ) ; external ;   { the ID string }

function IsInst( IdString : IdsType ) : boolean ; external ;

procedure UnInst( PrcPtr : word ); external;   { reinstall TSR program }


var ATimes : integer;                      { number of TSR activations }

{**********************************************************************}
{* DispInit: creates a pointer to the video RAM                       *}
{* Input  : none                                                      *}
{* Output : none                                                      *}
{**********************************************************************}

procedure DispInit;

var Regs: Registers;                  { stores the processor registers }

begin
  Regs.ah := $0f;              { function no. 15 = read the video mode }
  Intr($10, Regs);                     { call the BIOS video interrupt }
  if Regs.al=7 then                           { monochrome video card? }
    VioPtr := @MBuf     { yes, set pointer to the monochrome video RAM }
  else                                 { it's an EGA, VGA, or CGA card }
    VioPtr := @CBuf;                  { set pointer to color video RAM }
end;

{**********************************************************************}
{* SaveScreen: saves the screen contents in a buffer                  *}
{* Input  : SPTR : pointer to a buffer in which the screen contents   *}
{*                 will be saved                                      *}
{* Output : none                                                      *}
{**********************************************************************}

procedure SaveScreen( SPtr : VPtr );

var line,                                           { the current line }
    column : byte;                                { the current column }

begin
  for line:=1 to 25 do               { run through the 25 screen lines }
    for column:=1 to 80 do         { run through the 80 screen columns }
      SPtr^[line, column] := VioPtr^[line, column];   { save ch.&attr. }

end;

{**********************************************************************}
{* RestoreScreen: copies the contents of a buffer into the video RAM  *}
{* Input  : BPTR : pointer to the buffer whose contents are to be     *}
{*                 copied into the video RAM                          *}
{* Output : none                                                      *}
{**********************************************************************}

procedure RestoreScreen( BPtr : VPtr );

var line,                                           { the current line }
    column : byte;                                { the current column }

begin
  for line:=1 to 25 do               { run through the 25 screen lines }
    for column:=1 to 80 do         { run through the 80 screen columns }
      VioPtr^[line, column] := BPtr^[line, column];  { get ch. & attr. }
end;

{**********************************************************************}
{* ResPara: calculates the number of paragraphs which must be         *}
{*          allocated for the program                                 *}
{* Input  : none                                                      *}
{* Output : the number of paragraphs to be reserved                   *}
{**********************************************************************}

function ResPara : word;

begin
  ResPara := Seg(FreePtr^)+$1000-PrefixSeg;     { number of paragraphs }
end;

{**********************************************************************}
{* EndProc: Called by the assembler module when the TSR program is    *}
{*          reinstalled                                               *}
{* Input  : none                                                      *}
{* Output : none                                                      *}
{* Info   : This procedure must be in the main program and may not    *}
{*          be turned into a FAR procedure by the $F+ compiler        *}
{*          directive.                                                *}
{**********************************************************************}

{$F-}                                     { don't make a FAR procedure }

procedure EndProc;

begin
  TextBackground( Black );                           { dark background }
  TextColor( LightGray );                                 { light text }
  writeln('The TSR program was called ', ATimes, ' times.');
end;

{**********************************************************************}
{* Tsr: This procedure is called by the assembler module after the    *}
{*      hotkey is pressed.                                            *}
{* Input  : none                                                      *}
{* Output : none                                                      *}
{* Info   : This procedure must be in the main program and may not    *}
{*          be turned into a FAR procedure by the $F+ compiler        *}
{*          directive.                                                *}
{**********************************************************************}

{$F-}                                     { don't make a FAR procedure }

procedure Tsr;

var BufPtr : VPtr;            { stores pointer to the allocated blocks }
    Column,                                { the current screen column }
    Line  : byte;                            { the current screen line }
    Key  : char;

begin
  inc( ATimes );                              { increment call counter }
  DispInit;                       { determine address of the video RAM }
  GetMem(BufPtr, SizeOf(VBuf) );                     { allocate buffer }
  SaveScreen( BufPtr );                	    { save the screen contents }
  Line := WhereY;                            { get current screen line }
  Column := WhereX;                        { get current screen column }
  TextBackground( LightGray );                      { light background }
  TextColor( Black );                                      { dark text }
  ClrScr;                                     { clear the whole screen }
  GotoXY(22, 12);
  write('TSRP  -  (c) 1988 by MICHAEL TISCHER');
  GotoXY(30, 14);
  write('Please press a key...');
  Key := ReadKey;                                     { wait for a key }
  RestoreScreen( BufPtr );         { copy the old screen contents back }
  FreeMem( BufPtr, SizeOf(VBuf) );          { release allocated buffer }
  GotoXY( Column, Line );           { cursor back to original position }
end;

{**********************************************************************}
{**                           MAIN PROGRAM                           **}
{**********************************************************************}

begin
  writeln('TSRP  -  (c) 1988 by MICHAEL TISCHER');
  IdString := 'TROTZKY';
  if ( IsInst( IdString ) ) then          { program already installed? }
    begin                                                        { YES }
      writeln('The TSR program now disabled.');
      UnInst( Ofs( EndProc ) );                   { remove the program }

      {** if no end function is to be called, the call is: ************
       ** UnInst( NO_END_FTN );                            ************}
    end
  else                              { the program is not installed yet }
    begin
      ATimes := 0;                 { the program was not activated yet }
      writeln('TSR program now enabled.  Start: <LSHIFT> + ',
              '<RSHIFT>');
      TsrInit( Ofs(Tsr), LSHIFT or RSHIFT, ResPara, IdString );
    end;
end.
