/**********************************************************************/
/*                            E G A V G A C                           */
/*--------------------------------------------------------------------*/
/*    Task           : Demonstration using the functions available    */
/*                     in the EGA-/VGA-BIOS                           */
/*--------------------------------------------------------------------*/
/*    Author         : MICHAEL TISCHER                                */
/*    Developed on   : 08/30/1988                                     */
/*    Last update    : 05/02/1989                                     */
/*--------------------------------------------------------------------*/
/*    (MICROSOFT C)                                                   */
/*    Creation       : CL /AS /c EGAVGAC.C                            */
/*                     LINK EGAVGAC EGAVGACA;                         */
/*    Call           : EGAC                                           */
/*--------------------------------------------------------------------*/
/*    (BORLAND TURBO C)                                               */
/*    Creation       : Make a project file containing the following:  */
/*                       EGAVGAC                                      */
/*                       EGAVGACA.OBJ                                 */
/*                     Before compiling, select the Options menu      */
/*                     and the Compiler option - make sure that the   */
/*                     Small model is active                          */
/*                     Select the Linker option - make sure that the  */
/*                     Case-sensitive link is set to Off              */
/*                     The program will compile with one warning...   */
/*                     this is okay, it will run correctly            */
/**********************************************************************/

/*== Add include files ===============================================*/

#include <dos.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <bios.h>

/*== Typedefs ========================================================*/

typedef unsigned char BYTE;                          /* Create a byte */
typedef unsigned int WORD;
typedef BYTE BOOL;                          /* like BOOLEAN in Pascal */
typedef struct velb far * VP;    /* VP = FAR pointer to the video RAM */

/*== Function definition from the assembler module ===================*/

extern void chardef( BYTE ascii, BYTE table, BYTE lines,
                     BYTE amount, BYTE far * buf );

/*== Structures ======================================================*/

struct velb {          /* Describes a two-byte position on the screen */
             BYTE ascii_code,                              /* ASCII code */
                  attribute;                /* Corresponding attribute */
            };

/*== Macros ==========================================================*/

/*-- MK_FP creates a FAR pointer to an object out of a ---------------*/
/*-- segment address and an offset address ---------------------------*/

#ifndef MK_FP                               /* MK_FP not defined yet? */
#define MK_FP(seg, ofs) ((void far *) ((unsigned long) (seg)<<16|(ofs)))
#endif

#define VOFS(x,y) ( 80 * ( y ) + ( x ) )   /* Offsetpos. in video RAM */
#define VPOS(x,y) (VP) ( vptr + VOFS( x, y ) )     /* Pointer in VRAM */
#define GETCZ()   (vline)         /* Returns the current cursor line */
#define GETCS()   (vcolumn)      /* Returns the current cursor column */

/*== Constants =======================================================*/

#define TRUE  ( 1 == 1 )           /* Constants for working with BOOL */
#define FALSE ( 1 == 0 )

#define VIDEO_INT 0x10                        /* BIOS video interrupt */

#define MONO  0                           /* Monitor types for GETMON */
#define COLOR 1
#define EGA   2

#define PAUSE 100

/*== Global variables ================================================*/

VP vptr;               /* Pointer to the first character in video RAM */
BYTE vline,                    /* States the current cursor position */
     vcolumn;
BOOL mono;               /* TRUE if a monochrome monitor is connected */

/***********************************************************************
*  Function         : C E M U L                                        *
**--------------------------------------------------------------------**
*  Task             : Enables/disables cursor emulation on the         *
*                     EGA card.                                        *
*  Input parameters : - DOIT = TRUE : Emulation on                     *
*                              FALSE: Emulation off                    *
*  Return values    : None                                             *
***********************************************************************/

void cemul( BOOL doit )

{
 /*-- Definition of video info byte at offset address 0x87 within ----*/
 /*-- the BIOS variable segment --------------------------------------*/

 #define VIO_INFO_BYTE ((BYTE far *) MK_FP(0x40, 0x87))

 if ( doit )                             /* Cursor emulation enabled? */
  *VIO_INFO_BYTE |= 1;                              /* YES, set bit 0 */
 else                                                          /* NO, */
  *VIO_INFO_BYTE &= 254;                               /* clear bit 0 */
}

/***********************************************************************
*  Function         : G E T M O N                                      *
**--------------------------------------------------------------------**
*  Task             : Determines the type of monitor connected.        *
*  Input parameters : None                                             *
*  Return values    : Monitor type                                     *
*                       MONO = monochrome monitor                      *
*                       COLOR= Color monitor                           *
*                       EGA  = EGA or multisync monitor                *
***********************************************************************/

BYTE getmon()
 {
  union REGS regs;           /* Processor register for interrupt call */

  regs.h.ah = 0x12;       /* Function number: Determine configuration */
  regs.h.bl = 0x10;                            /* Sub-function number */
  int86(VIDEO_INT, &regs, &regs);        /* Call BIOS video interrupt */
  if ( regs.h.cl == 0x0B )                     /* Monochrome monitor? */
   return( MONO );                                             /* YES */
  if ( regs.h.cl == 0x08 )                          /* color monitor? */
   return( COLOR );                                            /* YES */
  else                                             /* NO, must be EGA */
   return( EGA );
 }

/***********************************************************************
*  Function         : S E T C U R                                      *
**--------------------------------------------------------------------**
*  Task             : Sets the screen cursor and the internal          *
*                     position of the output.                          *
*  Input parameters : - COLUMN   = the cursor column                   *
*                     - LINE     = the cursor line                     *
*  Return values    : None                                             *
***********************************************************************/

void setcur(BYTE column, BYTE line)
 {
  union REGS regs;           /* Processor register for interrupt call */

  regs.h.ah = 2;                                   /* Function number */
  regs.h.bh = 0;                               /* Use video page zero */
  regs.h.dh = vline = line; /* Use global variables for coordinates */
  regs.h.dl = vcolumn = column;
  int86(VIDEO_INT, &regs, &regs);        /* Call BIOS video interrupt */
 }

/***********************************************************************
*  Function         : S E T C O L                                      *
**--------------------------------------------------------------------**
*  Task             : Defines the contents of one of the 16 EGA        *
*                     color registers.                                 *
*  Input parameters : - REGNR = Color register number                  *
*                     - COLOR = Color value (0-15)                     *
*  Return values    : None                                             *
***********************************************************************/

void setcol(BYTE regnr, BYTE color)

{
 union REGS regs;        /* Processor register for the interrupt call */

 regs.h.ah = 0x10;               /* Function no.: Set color/attribute */
 regs.h.al = 0;                                     /* Sub-function 0 */
 regs.h.bl = regnr;                            /* Set register number */
 regs.h.bh = color & 63;         /* Set color number ( Bits 6 and 7 ) */
 int86(VIDEO_INT, &regs, &regs);  	 /* Call BIOS video interrupt */
}

/***********************************************************************
*  Function         : S E T B O R D E R                                *
**--------------------------------------------------------------------**
*  Task             : Sets the border color.                           *
*  Input  parameters: - COLOR = Color value (0-15)                     *
*  Return values      : None                                           *
***********************************************************************/

void setborder( BYTE color )

{
 union REGS regs;        /* Processor register for the interrupt call */

 regs.h.ah = 0x10;               /* Function no.: Set color/attribute */
 regs.h.al = 1;                                     /* Sub-function 1 */
 regs.h.bh = color & 15;                           /* Set color value */
 int86(VIDEO_INT, &regs, &regs);         /* Call BIOS video interrupt */
}

/***********************************************************************
*  Function         : S E T L I N E S                                  *
**--------------------------------------------------------------------**
*  Task             : Determines the number of lines.                  *
*  Input  parameters: - Sub-function no. for calling function 11(h)    *
*                        0x11 : 8*14 character set                     *
*                        0x12 : 8*8  character set                     *
*                        0x14 : 8*16 character set (VGA only)          *
*  Return values      : None                                           *
***********************************************************************/

void setlines( BYTE lines )

{
 union REGS regs;        /* Processor register for the interrupt call */

 regs.h.ah = 0x11;               /* Function no.: Character generator */
 regs.h.al = lines;                               /* Sub-function no. */
 regs.h.bl = 0;                              /* Use character table 0 */
 int86(VIDEO_INT, &regs, &regs);         /* Call BIOS video interrupt */
}

/***********************************************************************
*  Function         : I S _ E G A                                      *
**--------------------------------------------------------------------**
*  Task             : Determines whether an EGA card is installed.     *
*  Input  parameters: None                                             *
*  Return values      TRUE when an EGA or VGA card is installed, and   *
*                     false in any other case                          *
***********************************************************************/

BOOL is_ega()
 {
  union REGS regs;       /* Processor register for the interrupt call */

  regs.h.ah = 0x12; /* Function number: Determine video configuration */
  regs.h.bl = 0x10;                            /* Sub-function number */
  int86(VIDEO_INT, &regs, &regs);        /* Call BIOS video interrupt */
  if ( regs.h.bl != 0x10 )               /* Is it an EGA or VGA card? */
   /*-- Set pointer in video RAM for attached monitor ----------------*/
   vptr = (VP) MK_FP( (mono = regs.h.bh) ? 0xb000 : 0xb800, 0);
  return( regs.h.bl != 0x10 );           /* BL != 0x10 --> EGA or VGA */
 }

/***********************************************************************
*  Function         : I S _ V G A                                      *
**--------------------------------------------------------------------**
*  Task             : Determines whether a VGA card is installed.      *
*  Input  parameters: None                                             *
*  Return values    : TRUE when a VGA card is installed;               *
*                     FALSE in any other case.                         *
*  Info             : This function should be called before the        *
*                     is_ega function, because the parameters in the   *
*                     is_ega function also apply to VGA cards (i.e.,   *
*                     TRUE will be returned to is_ega for a VGA card.  *
*                     Call is_vga first in your own applications,      *
*                     then call is_ega.                                *
***********************************************************************/

BOOL is_vga()
 {
  union REGS regs;       /* Processor register for the interrupt call */

  regs.h.ah = 0x1A;    /* Function no.: Determine video configuration */
  regs.h.al = 0x00;                            /* Sub-function number */
  int86(VIDEO_INT, &regs, &regs);        /* Call BIOS video interrupt */
  if ( regs.h.al == 0x1A && ( regs.h.bl==7 || regs.h.bl==8 ) )
   {                            /* VGA card connected to VGA monitor? */
    mono = FALSE;
    vptr = (VP) MK_FP( 0xb800, 0 );       /* Set pointer in video RAM */
    return TRUE;
   }
  return FALSE;                              /* No VGA card installed */
 }

/***********************************************************************
*  Function         : P R I N T A T                                    *
**--------------------------------------------------------------------**
*  Task             : Displays a string on the screen.                 *
*                                                                      *
*  Input  parameters: - COLUMN   = Display column.                     *
*                     - LINE     = Display line.                       *
*                     - CHCOLOR  = Character attribute.                *
*                     - STRING   = Pointer to string.                  *
*  Return values      : None                                           *
*  Information      : - This function does not recognize format specs  *
*                       as supplied by PRINTF.                         *
*                     - When the function reaches the end of the       *
*                       screen, the screen will not scroll up.         *
***********************************************************************/

void printat(BYTE column, BYTE line, BYTE chcolor, char * string)

 {
  register VP lptr;                  /* Floating pointer to video RAM */
  register BYTE i;              /* points to the number of characters */
  unsigned newofs;                        /* Computes new coordinates */

  lptr = VPOS(column, line);             /* Set pointer in video RAM */
  for (i=0 ; *string ; ++lptr, ++i)                 /* execute string */
   {
    lptr->ascii_code = *(string++);            /* Character in video RAM */
    lptr->attribute = chcolor;                /* Set character attribute */
   }
  /*-- Compute new cursor coordinates --------------------------------*/

  vcolumn = (newofs = ((unsigned) line * 80 + column + i)) % 80;
  vline = newofs / 80;
 }

/***********************************************************************
*  Function         : P R I N T F A T                                  *
**--------------------------------------------------------------------**
*  Task             : Displays a string on the screen (like PRINTF),   *
*                     writing the string directly to video RAM.        *
*  Input  parameters: - COLUMN = Display column.                       *
*                     - LINE   = Display line.                         *
*                     - CHCOLOR= Character color.                      *
*                     - STRING = Pointer to the string.                *
*                     - ...    = Additional arguments as needed.       *
*  Return values      : None                                           *
*  Information      : - When the end of the screen is reached, the     *
*                       screen will not scroll up.                     *
*                   : - Der string can use the normal format specifier *
*                       group as used with PRINTF.                     *
***********************************************************************/

void printfat(BYTE column, BYTE line, BYTE chcolor, char * string,...)

 {
  va_list parameter;    /* Take parameter list for VA_... Macros from */
  char output[255];               /* the formatted, displayed string */

  va_start(parameter, string);/* Get parameters in PARAMETER variable */
  vsprintf(output, string, parameter);             /* Convert string */
  printat(column, line, chcolor, output);           /* Display string */
 }

/***********************************************************************
*  Function         : B L I N K I N G                                  *
**--------------------------------------------------------------------**
*  Task             : Defines the meaning of bit 7 of the attribute    *
*                     byte of a character in text mode.                *
*  Input  parameters: DOBLINK = TRUE  : Blink.                         *
*                               FALSE : Light background color.        *
*  Return values      : none                                           *
***********************************************************************/

void blinking( BOOL doblink )
 {
  union REGS regs;       /* Processor register for the interrupt call */

  regs.h.ah = 0x10;              /* Function no.: Set color/attribute */
  regs.h.al = 0x3;                             /* Sub-function number */
  regs.h.bl = doblink ? 1 : 0;                   /* BL = 1 : blinking */
  int86(VIDEO_INT, &regs, &regs);        /* Call BIOS video interrupt */
 }

/***********************************************************************
*  Function         : C L S                                            *
**--------------------------------------------------------------------**
*  Task             : Clears the screen and resets the video mode.     *
*                     This reset includes the palette registers, as    *
*                     well as the character set in use.                *
*  Input  parameters: none                                             *
*  Return values    : none                                             *
***********************************************************************/

void cls()
 {
  union REGS regs;       /* Processor register for the interrupt call */

  regs.h.ah = 0x0;                    /* Function no.: Set video mode */
  regs.h.al = ( mono ) ? 7 : 3;               /* 80x25-char text mode */
  int86(VIDEO_INT, &regs, &regs);        /* Call BIOS video interrupt */
 }

/***********************************************************************
*  Function         : N O K E Y                                        *
**--------------------------------------------------------------------**
*  Task             : Tests for a depressed key.                       *
*  Input  parameters: none                                             *
*  Return values    : TRUE if a key is depressed, otherwise            *
*                     FALSE.                                           *
***********************************************************************/

BOOL nokey()

{
#ifdef __TURBOC__                       /* Using TURBO C to compile?  */
 return( bioskey( 1 ) == 0 );         /* YES, read keyboard from BIOS */
#else                              /* Using Microsoft C to compile... */
 return( _bios_keybrd( _KEYBRD_READY ) == 0 );      /* Read from BIOS */
#endif
}

/***********************************************************************
*  Function         : E G A V G A                                      *
**--------------------------------------------------------------------**
*  Task             : Demonstrates the application of EGA/VGA BIOS     *
*                     functions                                        *
*  Input  parameters: VGA : TRUE when working with VGA card            *
*                           FALSE in any other case                    *
*  Return values    : none                                             *
***********************************************************************/

void egavga( BOOL VGA )
 {
  static BYTE font[120][14] = {      /* Character definition for logo */
   {  0,  0,255, 62, 28, 28, 28, 28, 28, 28, 28, 28, 28, 31},    /* T */
   {  0,  0,252,  7,  1,  1,  1,  1,  1,  1,  1,  1,  7,252},    /* h */
   {  0,  0,  0,  0,129,195,195,199,199,206,206,142, 14, 14},    /* e */
   {  0,  0, 62,193,128,128,  0,  0,  0,  0,  0,  0,  0,  0},    /* s */
   {  0,  0, 16,144,112, 48, 48, 16, 16,  0,  0,  0,  0,  0},    /* e */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /*   */
   {  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* l */
   {  0,  0,254,248,112,112,112,112,112,112,112,112,112,112},    /* i */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,252, 61, 30, 30, 28},    /* n */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,248,  6,  7,  3,  3},    /* e */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  0,  0,  0,128},    /* s */
   {  0,  0, 32, 96,224,224,224,224,224,254,224,224,224,224},    /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  6, 12, 28, 24},    /* c */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,240, 28,  6,  7,  7},    /* o */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 63, 15,  7,  7,  7},    /* n */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 30, 39, 71,135,128},    /* t */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,126, 30, 15, 15, 14},    /* a */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,124,131,  3,  1,  1},    /* i */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,129,131,195},    /* n */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 62,193,128,  0},    /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,192,224,224},    /* t */
   {  0,  0,248,120, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56},    /* h */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 31, 48, 48, 48, 48},    /* e */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,196, 52, 12,  4,  4},    /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* b */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* i */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* t */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* p */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* a */
   { 28, 28, 28, 28, 28, 28, 28, 28, 62,255,  0,  0,  0,  0},    /* t */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0,  0},    /* t */
   { 14, 14, 14,  7,  7,  3,  3,  1,  0,  0,  0,  0,  0,  0},    /* e */
   {  0,  0,  0,  0,  0,  0,128,128,193, 62,  0,  0,  0,  0},    /* r */
   {  0,  0,  0,  0, 16, 16, 32, 64,128,  0,  0,  0,  0,  0},    /* n */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* s */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0},    /*   */
   {112,112,112,112,112,112,112,112,248,254,  0,  0,  0,  0},    /* f */
   { 28, 28, 28, 28, 28, 28, 28, 28, 62,255,  0,  0,  0,  0},    /* o */
   {  3,  3,  3,  3,  3,  3,  3,  3,  7,159,  0,  0,  0,  0},    /* r */
   {128,128,128,128,128,128,128,128,192,240,  0,  0,  0,  0},    /*   */
   {224,224,224,224,224,224, 96,112, 49, 30,  0,  0,  0,  0},    /* t */
   { 56, 63, 56, 56, 56, 24, 92, 76,134,  1,  0,  0,  0,  0},    /* h */
   {  7,255,  0,  0,  0,  0,  1,  2, 12,240,  0,  0,  0,  0},    /* e */
   {  7,  7,  7,  7,  7,  7,  7,  7, 15, 63,  0,  0,  0,  0},    /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,128,224,  0,  0,  0,  0},    /* c */
   { 14, 14, 14, 14, 14, 14, 14, 14, 31,127,  0,  0,  0,  0},    /* h */
   {  1,  1,  1,  1,  1,  1,  1,  1,  3,207,  0,  0,  0,  0},    /* a */
   {192,192,192,193,193,195,195,193,225,248,  0,  0,  0,  0},    /* r */
   {  0,  7,120,192,192,128,128,192,195,124,  0,  0,  0,  0},    /* a */
   {224,224,224,224,224,224,224,240,112, 29,  0,  0,  0,  0},    /* c */
   { 56, 56, 56, 56, 56, 56, 56, 56,124,255,  0,  0,  0,  0},    /* t */
   { 31, 31, 31,  0,  0, 64, 96, 96,112, 71,  0,  0,  0,  0},    /* e */
   {  0,224,248,252, 28, 12,  4, 12, 24,224,  0,  0,  0,  0},    /* r */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* s */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* n */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* e */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* e */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* d */
   {  0,  0,252, 60, 30, 30, 30, 23, 23, 23, 19, 19, 19, 17},    /* e */
   {  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,130,130,130,196},    /* d */
   {  0,  0,126,120,240,240,240,112,112,112,112,112,112,112},    /*   */
   {  0,  0, 28, 28, 28,  0,  0,  0,  0,252, 60, 28, 28, 28},    /* i */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  6, 12, 28, 24},    /* n */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,240, 12,  2,  7,  7},    /*   */
   {  0,  0, 63, 15,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7},    /* t */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 62, 65,129,128,  0},    /* h */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,192,192,224},    /* e */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 63, 64,224,224,224},    /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,192, 96,112,112},    /* l */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  7, 24, 48,112, 96},    /* o */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,192,112, 24, 28, 28},    /* g */
   {  0,  0,252, 60, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28},    /* o */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},    /* a */
   {  0,  0, 63, 56, 48, 48, 32, 32, 32,  0,  0,  0,  0,  0},    /* t */
   {  0,  0,255,112,112,112,112,112,112,112,112,112,112,112},    /*   */
   {  0,  0,225,225, 97, 32, 32, 32, 32, 15,  3,  1,  1,  1},    /* t */
   {  0,  0,192,192,192,  0,  0,  0,  0,192,193,195,195,195},    /* h */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,252,  3,  0,  0,  0},    /* e */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 64, 65,195, 71, 70},    /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,124,131,  0,  1,  1},    /* c */
   {  0,  0, 15,  3,  1,  1,  1,  1,  1,  1,  1,129,193,193},    /* e */
   {  0,  0,192,192,192,192,192,192,192,207,208,224,224,192},    /* n */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,128, 96,112, 48, 56},    /* t */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  3, 12, 24, 56, 48},    /* e */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,224, 56, 12, 14, 14},    /* r */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,126, 30, 14, 15, 15},    /*   */
   {  0,  0,  0,  0,  0,  0,  0,  0,  0, 60, 78,142, 14,  0},    /* o */
   { 17, 17, 16, 16, 16, 16, 16, 16, 48,254,  0,  0,  0,  0},    /* f */
   {196,196,232,232,232,112,112, 80, 32, 35,  0,  0,  0,  0},    /*   */
   {112,112,112,112,112,112,112,112,248,254,  0,  0,  0,  0},    /* t */
   { 28, 28, 28, 28, 28, 28, 28, 28, 62,255,  0,  0,  0,  0},    /* h */
   { 56, 56, 56, 56, 56, 24, 28, 12,  6,129,  0,  0,  0,  0},    /* e */
   {  7,  0,  0,  0,  0,  0,  1,  2, 12,240,  0,  0,  0,  0},    /*   */
   {  7,  7,  7,  7,  7,  7,  7,  7, 15, 63,  0,  0,  0,  0},    /* s */
   {  0,  0,  0,  0,  0,  0,  0,  0,129,231,  0,  0,  0,  0},    /* c */
   {224,224,224,224,224,225,225,224,240,252,  0,  0,  0,  0},    /* r */
   {  0,  3, 60,224,224,192,192,224,225, 62,  0,  0,  0,  0},    /* e */
   {112,240,112,112,112,112,112,120,184, 14,  0,  0,  0,  0},    /* e */
   {224,255,224,224,224, 96,112, 48, 24,  7,  0,  0,  0,  0},    /* n */
   { 28,252,  0,  0,  0,  0,  4,  8, 48,192,  0,  0,  0,  0},    /*   */
   { 28, 28, 28, 28, 28, 28, 28, 28, 62,255,  0,  0,  0,  0},
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0,  0},
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0},
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0},
   {112,112,112,112,112,112,112,112,248,254,  0,  0,  0,  0},
   {  1,  1,  1,  1,  1,  1,  1,  1,  3, 15,  0,  0,  0,  0},
   {193,193,192,192,192,194,195,195,227,250,  0,  0,  0,  0},
   {240,254,255, 15,  1,  0,  0,  0,129,126,  0,  0,  0,  0},
   { 14, 14,142,206,206,198, 71,195,129,  0,  0,  0,  0,  0},
   {  1,  0,  0,  0,  0,  0,  0,  0,131,124,  0,  0,  0,  0},
   {193,  1,  1,  1,  1,  1, 65,129,  3, 15,  0,  0,  0,  0},
   {192,192,192,192,192,192,192,192,224,249,  0,  0,  0,  0},
   { 56, 56, 56, 56, 56, 56, 56, 56,124,255,  0,  0,  0,  0},
   {112,127,112,112,112, 48, 56, 24, 12,  3,  0,  0,  0,  0},
   { 14,254,  0,  0,  0,  0,  2,  4, 24,224,  0,  0,  0,  0},
   { 14, 14, 14, 14, 14, 14, 14, 14, 31,127,  0,  0,  0,  0},
   {  0,  0,  0,  0,  0,  0,  0,  0,  0,192,  0,  0,  0,  0}
  };

  union REGS regs;       /* Processor register for the interrupt call */
  unsigned i, j, k;                                   /* Loop counter */
  double delay;                             /* Loop counter for PAUSE */

  /*-- Prepare screen ------------------------------------------------*/

  cls();                                             /* Clears screen */
  blinking( FALSE );    /* Light background color instead of blinking */
  setcur(0, 0);                   /* Move cursor to upper left corner */

  /*-- Install EGA and VGA hardcopy routine --------------------------*/
  regs.h.ah = 0x12;                 /* Function no.: Alternate Select */
  regs.h.bl = 0x20;     /* Sub-funct. 0x20 = Install hardcopy routine */
  int86(VIDEO_INT, &regs, &regs);        /* Call BIOS video interrupt */

  if ( VGA )                               /* Check for compatibility */
   {                                   /* and check custom characters */
    regs.h.ah = 0x12;                    /* VGA card in 350-line mode */
    regs.h.bl = 0x30;      			   /* Toggle EGA card */
    regs.h.al = 1;
    int86(VIDEO_INT, &regs, &regs);  	 /* Call BIOS video interrupt */

    setlines( 0x11 );                    /* Enable 8x14 character set */
   }

  chardef(128, 0, 14, 120, (BYTE far *) font);   /* Define characters */

  for (i=0; i<250; ++i)                     /* Execute loop 250 times */
   {                               /* Write color blocks to video RAM */
    printfat(GETCS(), GETCZ(), ((i % 14) + 1) << 4, "     ");
    printfat(GETCS(), GETCZ(), 0, "   ");
   }
  for (i=10; i<16; ++i)                    /* Allocate space for logo */
   printat(22, i, 0, "                                  ");
  for (k=128, i=0; i<4; ++i)            /* The logo consists of ASCII */
   {                                            /* characters 128-248 */
    for (j=0; j<30; ++j)
     printfat(j+24, i+11, 15, "%c", k++);
   }
  printat(1, 1, 15, "The most important characters are");
  printat(1, 2, 15, "still present despite the logo!  ");
  printat(1, 3, 15, "                                 ");
  printat(1, 4, 15, " !\"#$%&'()*+-./0123456789:;<=>?@ ");
  printat(1, 5, 15, " ABCDEFGHIJKLMNOPQRSTUVXXYZ[\\]^_ ");
  printat(1, 6, 15, " `abcdefghijklmnopqrstuvxyz{|}~  ");
  printat(33, 21, 15, "                                           ");
  printat(33, 22, 15, " Press any key to end the program.         ");
  printat(33, 23, 15, "                                           ");
  setcur( 34, 22);

  /*-- Change colors in the color blocks -----------------------------*/

  i = 0;                         /* Starting value for color register */
  while ( nokey() )            /* Repeat until the user presses a key */
   {
    for ( delay=0.0; delay < PAUSE; ++delay )
     ;
    ++i;              /* Increment color value for the first register */
    for (j=1; j<15; ++j)              /* Go through registers 1 to 14 */
     {
      setcol(j, i+j & 63);           /* Write color value in register */
      if ( !nokey() )                                 /* Key pressed? */
       break;                  /* YES --> Stop loop before restarting */
     }
   }

  if ( VGA )                                 /* Go into 400 line mode */
   {                         /* Enable VGA card                       */
    regs.h.ah = 0x12;
    regs.h.bl = 0x30;
    regs.h.al = 2;
    int86(VIDEO_INT, &regs, &regs);      /* Call BIOS video interrupt */

    setlines( 0x14 );                    /* Enable 8*16 character set */
   }

  cls();                                              /* Clear screen */
 }

/**********************************************************************/
/**                           MAIN  PROGRAM                          **/
/**********************************************************************/

void main()
{
 if ( is_vga() )                    /* Is there a VGA card installed? */
   egavga( TRUE );                                             /* YES */
 else                                     /* No VGA installed - go on */
  {
   if ( is_ega() )                 /* Is there an EGA card installed? */
    {                                                          /* YES */
     if ( getmon() == EGA )     /* Is there an EGA monitor connected? */
      egavga( FALSE );                             /* YES, start demo */
     else
      {                                              /* wrong monitor */
       printf("This program functions only with an\n");
       printf("EGA monitor.                       \n");
      }
    }
   else                           /* If no EGA or VGA card connected  */
    printf( "ATTENTION! There is neither an EGA nor a "
            " VGA card installed.\n" );
  }
}
